﻿using System;
using System.Threading;
using System.Windows.Forms;
using Psion.RFID.HF;
using Psion.RFID;
using PsionRFIDDemo.ExecManager;

namespace HF_Demo
{
    internal static class Program
    {
        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [MTAThread]
        private static void Main()
        {
            // get device information to check if the demo is running on EP10 or WAP4
            string oem = Device.OEMInfo;
#if EP10
            if (oem.IndexOf("7515") == -1) // is not a EP10
            {
                MessageBox.Show(string.Format("This device is not a EP10. It will not be able to start this demo. Please check and get the good installer. Current device is {0}", oem));
#else
            // WAP4
            if (oem.IndexOf("7528") == -1) // is not a WAP4
            {
                MessageBox.Show(
                    string.Format(
                        "This device is not a WAP4. It will not be able to start this demo. Please check and get the good installer. Current device is {0}",
                        oem));
#endif
                return; // exit app
            }

            // check if Keyboard wedge is not already running
            string appName = "HF Keyboard Wedge";
            using (var execMgr = new AppExecutionManager(appName))
            {
                if (!execMgr.IsFirstInstance)
                {
                    MessageBox.Show(appName + " is running, you should close it before to start the HF Demo", "Reminder", MessageBoxButtons.OK,
                                    MessageBoxIcon.Asterisk, MessageBoxDefaultButton.Button1);
                    return; // exit app
                }
            }

            // check if the Demo is not already running
            appName = "HF Demo";
            using (var execMgr = new AppExecutionManager(appName))
            {
                if (!execMgr.IsFirstInstance)
                {
                    MessageBox.Show(appName + " already running", "Reminder", MessageBoxButtons.OK,
                                    MessageBoxIcon.Asterisk, MessageBoxDefaultButton.Button1);
                    return; // exit app
                }

                Reader _reader;
                RFIDDriver _rfidDriver;

#if EP10
            try
            {
                // Instanciate reader
                //  throw exception if rfid driver is not installed
                //  throw exception if rfid driver is not activated
                _reader = new PsionDriverReader();  

                // open reader
                _reader.Open();
 
                var data = _reader.GetVersion();
                if(data.Length == 0)
                    throw new Exception("Communication error");
            }
            catch (Exception exception)
            {
                if (exception.Message.Equals("Psion RFID Driver is not installed"))
                    MessageBox.Show(exception.Message + "\r\n\r\nPlease install the RFID driver CAB file before to execute the demo");
                else
                    MessageBox.Show(exception.Message);
                return;
            }
#else
#if WAP4
                // Instanciate driver
                _rfidDriver = new RFIDDriver();

                try
                {
                    // Check if the RFID driver is well installed
                    if (!_rfidDriver.IsInstalled)
                        throw new Exception("Please install the RFID driver CAB file before to execute the demo");

                    // Enable driver
                    //  - Turn ON the RFID reader
                    //  - Assign the com port number
                    _rfidDriver.Enable();

                    // Necessary sleep to keep the USB mounting
                    Thread.Sleep(250);

                    // Instanciate reader
                    _reader = new SerialReader();

                    // Open reader
                    string comPort = string.Empty;
                    if (_rfidDriver.ComPort < 10) comPort = String.Format("COM{0}", _rfidDriver.ComPort);
                    else comPort = String.Format("$device\\COM{0}", _rfidDriver.ComPort);

                    ((SerialReader) _reader).Open(
                        comPort,
                        115200);

                    var data = _reader.GetVersion();
                    if (data.Length == 0)
                        throw new Exception("Communication error");
                }
                catch (Exception exception)
                {
                    MessageBox.Show(exception.Message);
                    return;
                }
#else
#error Please select EP10 or WAP4 configuration
#endif
#endif

                SettingsForm settingsForm = null;
                try
                {
                    // This form gives access to Operation Mode Register (EEPROM) of RFID reader
                    //  It allows to set only the kind of tag that you are focusing on 
                    settingsForm = new SettingsForm(_reader);
                    settingsForm.ShowDialog();
                }
                catch (Exception exception)
                {
                    MessageBox.Show(exception.Message);
                    return;
                }
                finally
                {
                    if (settingsForm != null) settingsForm.Dispose();
                }

                // Go to main form with access to all the others forms
                Application.Run(new MainForm(_reader));
            }
        }

    }
}